%%
%% %CopyrightBegin%
%%
%% Copyright Ericsson AB 2011-2024. All Rights Reserved.
%%
%% Licensed under the Apache License, Version 2.0 (the "License");
%% you may not use this file except in compliance with the License.
%% You may obtain a copy of the License at
%%
%%     http://www.apache.org/licenses/LICENSE-2.0
%%
%% Unless required by applicable law or agreed to in writing, software
%% distributed under the License is distributed on an "AS IS" BASIS,
%% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%% See the License for the specific language governing permissions and
%% limitations under the License.
%%
%% %CopyrightEnd%
%%

-module(pubkey_ocsp_SUITE).

-include_lib("common_test/include/ct.hrl").
-include_lib("public_key/include/public_key.hrl").
%% Note: This directive should only be used in test suites.
-compile([export_all, nowarn_export_all]).

-define(NONCE,
        <<4,8,93,33,1,52,187,3,12,142>>).
-define(OCSP_RESPONSE_DER,
        <<48,130,7,36,10,1,0,160,130,7,29,48,130,7,25,6,9,43,6,1,5,5,7,48,1,1,4,130,7,10,48,130,7,6,48,130,1,10,161,129,134,48,129,131,49,14,48,12,6,3,85,4,3,12,5,111,116,112,67,65,49,19,48,17,6,3,85,4,11,12,10,69,114,108,97,110,103,32,79,84,80,49,20,48,18,6,3,85,4,10,12,11,69,114,105,99,115,115,111,110,32,65,66,49,11,48,9,6,3,85,4,6,19,2,83,69,49,18,48,16,6,3,85,4,7,12,9,83,116,111,99,107,104,111,108,109,49,37,48,35,6,9,42,134,72,134,247,13,1,9,1,22,22,112,101,116,101,114,64,101,114,105,120,46,101,
          114,105,99,115,115,111,110,46,115,101,24,15,50,48,50,51,48,55,50,48,49,50,50,57,52,57,90,48,81,48,79,48,58,48,9,6,5,43,14,3,2,26,5,0,4,20,227,147,252,182,155,101,129,45,194,162,22,93,127,46,112,193,196,28,241,232,4,20,34,25,129,87,115,255,155,246,200,97,92,7,51,110,152,61,97,155,164,171,2,1,9,128,0,24,15,50,48,50,51,48,55,50,48,49,50,50,57,52,57,90,161,27,48,25,48,23,6,9,43,6,1,5,5,7,48,1,2,4,10,4,8,93,33,1,52,187,3,12,142,48,13,6,9,42,134,72,134,247,13,1,1,11,5,0,3,130,1,1,0,182,228,165,33,
          173,232,46,125,152,237,37,120,186,223,188,231,181,61,72,89,210,75,38,182,146,218,223,53,38,104,100,89,19,79,48,159,109,70,25,34,143,253,199,92,162,248,245,240,61,82,39,44,243,148,109,21,112,74,206,246,5,146,12,175,235,170,225,206,115,148,109,1,216,194,98,149,105,106,232,83,95,229,196,237,246,6,177,167,177,52,242,63,56,93,74,92,28,246,138,137,99,28,239,68,130,184,182,140,50,206,10,166,243,118,11,153,79,178,220,166,161,45,83,90,58,152,78,229,27,54,147,125,106,199,192,182,242,242,98,69,94,148,
          163,130,154,168,134,24,7,211,176,133,1,156,206,22,197,139,59,66,110,121,187,101,221,16,11,114,106,56,178,21,3,189,89,233,228,8,127,150,247,151,97,145,204,117,80,120,174,191,12,150,148,180,86,16,86,82,170,184,185,10,182,252,210,66,238,144,233,244,94,144,143,200,0,172,179,194,109,137,186,95,180,111,242,81,132,254,70,174,168,9,181,145,154,180,7,78,17,56,24,195,236,80,46,1,241,0,52,172,84,154,43,117,130,153,160,130,4,224,48,130,4,220,48,130,4,216,48,130,3,192,160,3,2,1,2,2,1,1,48,13,6,9,42,134,
          72,134,247,13,1,1,11,5,0,48,129,134,49,17,48,15,6,3,85,4,3,12,8,101,114,108,97,110,103,67,65,49,19,48,17,6,3,85,4,11,12,10,69,114,108,97,110,103,32,79,84,80,49,20,48,18,6,3,85,4,10,12,11,69,114,105,99,115,115,111,110,32,65,66,49,18,48,16,6,3,85,4,7,12,9,83,116,111,99,107,104,111,108,109,49,11,48,9,6,3,85,4,6,19,2,83,69,49,37,48,35,6,9,42,134,72,134,247,13,1,9,1,22,22,112,101,116,101,114,64,101,114,105,120,46,101,114,105,99,115,115,111,110,46,115,101,48,30,23,13,50,51,48,55,50,48,49,50,50,57,
          52,52,90,23,13,51,51,48,53,50,56,49,50,50,57,52,52,90,48,129,131,49,14,48,12,6,3,85,4,3,12,5,111,116,112,67,65,49,19,48,17,6,3,85,4,11,12,10,69,114,108,97,110,103,32,79,84,80,49,20,48,18,6,3,85,4,10,12,11,69,114,105,99,115,115,111,110,32,65,66,49,11,48,9,6,3,85,4,6,19,2,83,69,49,18,48,16,6,3,85,4,7,12,9,83,116,111,99,107,104,111,108,109,49,37,48,35,6,9,42,134,72,134,247,13,1,9,1,22,22,112,101,116,101,114,64,101,114,105,120,46,101,114,105,99,115,115,111,110,46,115,101,48,130,1,34,48,13,6,9,42,
          134,72,134,247,13,1,1,1,5,0,3,130,1,15,0,48,130,1,10,2,130,1,1,0,224,126,14,188,23,118,17,120,117,2,151,131,96,96,81,128,52,8,219,111,48,52,116,100,176,30,58,84,47,206,254,180,60,112,107,216,0,213,16,164,57,110,181,88,62,78,208,125,127,255,36,56,29,37,165,108,64,115,95,46,212,88,172,92,20,177,247,84,13,124,142,41,248,131,77,91,114,228,157,125,20,155,64,70,60,206,180,54,12,80,153,230,208,10,189,163,140,22,146,156,99,21,213,182,98,144,208,94,71,69,249,100,109,104,226,169,55,210,194,244,111,92,
          16,87,6,93,42,174,62,243,175,60,139,134,67,116,107,208,120,214,103,12,182,22,229,195,60,133,235,228,230,17,0,101,185,117,3,52,252,75,173,144,14,30,36,132,251,47,0,161,159,183,230,201,93,76,82,202,66,213,34,44,88,148,92,240,21,152,135,102,160,212,173,100,225,227,143,229,74,120,120,147,179,228,80,127,246,118,213,141,122,209,29,48,47,188,130,179,111,83,234,203,22,0,167,15,53,193,172,80,117,62,98,63,233,161,102,96,21,10,72,179,86,205,96,124,62,133,9,28,197,39,190,23,72,49,2,3,1,0,1,163,130,1,80,
          48,130,1,76,48,15,6,3,85,29,19,1,1,255,4,5,48,3,1,1,255,48,11,6,3,85,29,15,4,4,3,2,1,6,48,29,6,3,85,29,14,4,22,4,20,34,25,129,87,115,255,155,246,200,97,92,7,51,110,152,61,97,155,164,171,48,129,198,6,3,85,29,35,4,129,190,48,129,187,128,20,196,35,212,128,180,55,39,81,140,96,141,212,14,41,206,56,214,196,133,175,161,129,140,164,129,137,48,129,134,49,17,48,15,6,3,85,4,3,12,8,101,114,108,97,110,103,67,65,49,19,48,17,6,3,85,4,11,12,10,69,114,108,97,110,103,32,79,84,80,49,20,48,18,6,3,85,4,10,12,11,
          69,114,105,99,115,115,111,110,32,65,66,49,18,48,16,6,3,85,4,7,12,9,83,116,111,99,107,104,111,108,109,49,11,48,9,6,3,85,4,6,19,2,83,69,49,37,48,35,6,9,42,134,72,134,247,13,1,9,1,22,22,112,101,116,101,114,64,101,114,105,120,46,101,114,105,99,115,115,111,110,46,115,101,130,20,43,178,6,127,40,48,98,119,68,20,137,15,178,249,179,119,3,205,202,41,48,33,6,3,85,29,17,4,26,48,24,129,22,112,101,116,101,114,64,101,114,105,120,46,101,114,105,99,115,115,111,110,46,115,101,48,33,6,3,85,29,18,4,26,48,24,129,
          22,112,101,116,101,114,64,101,114,105,120,46,101,114,105,99,115,115,111,110,46,115,101,48,13,6,9,42,134,72,134,247,13,1,1,11,5,0,3,130,1,1,0,2,39,149,225,217,60,119,245,177,96,137,169,49,169,0,163,164,218,40,228,230,193,142,138,206,47,0,176,80,99,16,226,79,124,237,230,91,57,148,49,250,208,42,193,106,53,86,163,205,254,245,90,45,240,172,107,162,160,59,7,246,245,41,106,44,239,47,78,197,79,123,55,217,87,181,221,73,88,47,122,30,195,225,6,28,237,49,250,105,85,214,69,86,243,73,81,101,192,31,250,31,
          55,111,63,11,1,147,63,144,241,132,32,161,92,168,152,19,29,233,88,234,4,134,144,26,70,162,219,31,125,205,202,94,45,111,3,17,66,62,208,17,188,179,94,222,238,248,79,102,80,138,217,80,233,100,152,240,11,81,36,130,175,152,182,221,2,26,24,33,180,242,63,33,223,18,131,11,52,51,1,193,24,222,91,47,100,131,173,32,69,159,13,94,246,193,182,127,242,164,131,112,92,179,65,79,235,174,161,194,201,255,119,2,251,215,203,135,16,154,55,69,82,33,69,60,223,118,35,56,22,228,106,80,57,180,62,124,121,244,121,197,123,
          242,190,55,26,32,214,176,53,28,117,171,162,76,160>>).
-define(ISSUER_CERT,
        {'OTPCertificate',{'OTPTBSCertificate',v3,1,
                           {'SignatureAlgorithm',{1,2,840,113549,1,1,11},'NULL'},
                           {rdnSequence,[[{'AttributeTypeAndValue',{2,5,4,3},{utf8String,<<"erlangCA">>}}],[{'AttributeTypeAndValue',{2,5,4,11},{utf8String,<<"Erlang OTP">>}}],[{'AttributeTypeAndValue',{2,5,4,10},{utf8String,<<"Ericsson AB">>}}],[{'AttributeTypeAndValue',{2,5,4,7},{utf8String,<<"Stockholm">>}}],[{'AttributeTypeAndValue',{2,5,4,6},"SE"}],[{'AttributeTypeAndValue',{1,2,840,113549,1,9,1},"peter@erix.ericsson.se"}]]},
                           {'Validity',{utcTime,"230720122944Z"},{utcTime,"330528122944Z"}},
                           {rdnSequence,[[{'AttributeTypeAndValue',{2,5,4,3},{utf8String,<<"otpCA">>}}],[{'AttributeTypeAndValue',{2,5,4,11},{utf8String,<<"Erlang OTP">>}}],[{'AttributeTypeAndValue',{2,5,4,10},{utf8String,<<"Ericsson AB">>}}],[{'AttributeTypeAndValue',{2,5,4,6},"SE"}],[{'AttributeTypeAndValue',{2,5,4,7},{utf8String,<<"Stockholm">>}}],[{'AttributeTypeAndValue',{1,2,840,113549,1,9,1},"peter@erix.ericsson.se"}]]},
                           {'OTPSubjectPublicKeyInfo',{'PublicKeyAlgorithm',{1,2,840,113549,1,1,1},'NULL'},
                            {'RSAPublicKey',28339541610808720697355110173776661351591299627563886520724840524156173758352096851973677497931775737934761046080070714946146799362954032218057107079242015447767834113919294749723734648362271942219510034694564267611067362221363996274084106982740657802072527220370548608128815474010110407985824336936761317435951968068792959040741225688753787656107492053262271328711261658605960823613199135109317800726834363235926476527282638847781232871073244485999722937494811217769991698287040924806827188498324716317026838449695477777978536565396172198805249805405552866638039988130119465502001541293012996469049417900821835106353,
                             65537}},
                           asn1_NOVALUE,asn1_NOVALUE,
                           [{'Extension',{2,5,29,19},true,{'BasicConstraints',true,asn1_NOVALUE}},
                            {'Extension',{2,5,29,15},false,[keyCertSign,cRLSign]},
                            {'Extension',{2,5,29,14},false,<<34,25,129,87,115,255,155,246,200,97,92,7,51,110,152,61,97,155,164,171>>},
                            {'Extension',{2,5,29,35},
                             false,
                             {'AuthorityKeyIdentifier',<<196,35,212,128,180,55,39,81,140,96,141,212,14,41,206,56,214,196,133,175>>,
                              [{directoryName,{rdnSequence,[[{'AttributeTypeAndValue',{2,5,4,3},{utf8String,<<"erlangCA">>}}],
                                                            [{'AttributeTypeAndValue',{2,5,4,11},{utf8String,<<"Erlang OTP">>}}],
                                                            [{'AttributeTypeAndValue',{2,5,4,10},{utf8String,<<"Ericsson AB">>}}],
                                                            [{'AttributeTypeAndValue',{2,5,4,7},{utf8String,<<"Stockholm">>}}],
                                                            [{'AttributeTypeAndValue',{2,5,4,6},"SE"}],
                                                            [{'AttributeTypeAndValue',{1,2,840,113549,1,9,1},"peter@erix.ericsson.se"}]]}}],
                              249456701733760087187851894331345805369320262185}},
                            {'Extension',{2,5,29,17},false,[{rfc822Name,"peter@erix.ericsson.se"}]},
                            {'Extension',{2,5,29,18},false,[{rfc822Name,"peter@erix.ericsson.se"}]}]},
         {'SignatureAlgorithm',{1,2,840,113549,1,1,11},'NULL'},
         <<2,39,149,225,217,60,119,245,177,96,137,169,49,169,0,163,164,218,40,228,230,193,142,138,206,47,0,176,80,99,16,226,79,124,237,230,91,57,148,49,250,208,42,193,106,53,86,163,205,254,245,90,45,240,172,107,162,160,59,7,246,245,41,106,44,239,47,78,197,79,123,55,217,87,181,221,73,88,47,122,30,195,225,6,28,237,49,250,105,85,214,69,86,243,73,81,101,192,31,250,31,55,111,63,11,1,147,63,144,241,132,32,161,92,168,152,19,29,233,88,234,4,134,144,26,70,162,219,31,125,205,202,94,45,111,3,17,
           66,62,208,17,188,179,94,222,238,248,79,102,80,138,217,80,233,100,152,240,11,81,36,130,175,152,182,221,2,26,24,33,180,242,63,33,223,18,131,11,52,51,1,193,24,222,91,47,100,131,173,32,69,159,13,94,246,193,182,127,242,164,131,112,92,179,65,79,235,174,161,194,201,255,119,2,251,215,203,135,16,154,55,69,82,33,69,60,223,118,35,56,22,228,106,80,57,180,62,124,121,244,121,197,123,242,190,55,26,32,214,176,53,28,117,171,162,76,160>>}
       ).
-define(RESPONDER_CERT, #cert{otp = ?ISSUER_CERT}).
-define(A_SERVER_CERT,
        {'OTPCertificate',{'OTPTBSCertificate',v3,9,
                           {'SignatureAlgorithm',{1,2,840,113549,1,1,11},'NULL'},
                           {rdnSequence,[[{'AttributeTypeAndValue',{2,5,4,3},{utf8String,<<"otpCA">>}}],[{'AttributeTypeAndValue',{2,5,4,11},{utf8String,<<"Erlang OTP">>}}],[{'AttributeTypeAndValue',{2,5,4,10},{utf8String,<<"Ericsson AB">>}}],[{'AttributeTypeAndValue',{2,5,4,6},"SE"}],[{'AttributeTypeAndValue',{2,5,4,7},{utf8String,<<"Stockholm">>}}],[{'AttributeTypeAndValue',{1,2,840,113549,1,9,1},"peter@erix.ericsson.se"}]]},
                           {'Validity',{utcTime,"230720133137Z"},{utcTime,"330528133137Z"}},
                           {rdnSequence,[[{'AttributeTypeAndValue',{2,5,4,3},{utf8String,<<"a.server">>}}],[{'AttributeTypeAndValue',{2,5,4,11},{utf8String,<<"Erlang OTP">>}}],[{'AttributeTypeAndValue',{2,5,4,10},{utf8String,<<"Ericsson AB">>}}],[{'AttributeTypeAndValue',{2,5,4,6},"SE"}],[{'AttributeTypeAndValue',{2,5,4,7},{utf8String,<<"Stockholm">>}}],[{'AttributeTypeAndValue',{1,2,840,113549,1,9,1},"peter@erix.ericsson.se"}]]},
                           {'OTPSubjectPublicKeyInfo',{'PublicKeyAlgorithm',{1,2,840,113549,1,1,1},'NULL'},
                            {'RSAPublicKey',21289228001579879261333836287004823332222603265803194664536540127079242843046558012638236832904402369325784443025964406623777246658832921141960060475804229950419927075126320682148878633948860170628231809668072735203197673472127901696952853229103382707195692260834407944629128675563945194876045436588557031550105450845916227210816469918126215406473272174435113136059751668904728646605321836951857481644376964329827326510385947455341932553729402879301386590388317914766820519616318357183422565964994898098302388155119056040428369731559112404176013041801429296771966314089067058782535413289830323744573459011210321571511,
                             65537}},
                           asn1_NOVALUE,asn1_NOVALUE,
                           [{'Extension',{2,5,29,19},false,{'BasicConstraints',false,asn1_NOVALUE}},
                            {'Extension',{2,5,29,15},false,[digitalSignature,nonRepudiation,keyEncipherment]},
                            {'Extension',{2,5,29,14},false,<<135,115,253,182,154,247,175,77,237,3,215,88,195,15,182,214,248,144,201,202>>},
                            {'Extension',{2,5,29,35},
                             false,
                             {'AuthorityKeyIdentifier',<<63,22,3,246,66,83,233,142,103,82,105,230,175,132,130,192,87,79,122,232>>,
                              [{directoryName,{rdnSequence,[[{'AttributeTypeAndValue',{2,5,4,3},{utf8String,<<"erlangCA">>}}],
                                                            [{'AttributeTypeAndValue',{2,5,4,11},{utf8String,<<"Erlang OTP">>}}],
                                                            [{'AttributeTypeAndValue',{2,5,4,10},{utf8String,<<"Ericsson AB">>}}],
                                                            [{'AttributeTypeAndValue',{2,5,4,7},{utf8String,<<"Stockholm">>}}],
                                                            [{'AttributeTypeAndValue',{2,5,4,6},"SE"}],
                                                            [{'AttributeTypeAndValue',{1,2,840,113549,1,9,1},"peter@erix.ericsson.se"}]]}}],
                              1}},
                            {'Extension',{2,5,29,17},false,[{dNSName,"host.example.com"}]},
                            {'Extension',{2,5,29,18},false,[{rfc822Name,"peter@erix.ericsson.se"}]}]},
         {'SignatureAlgorithm',{1,2,840,113549,1,1,11},'NULL'},
         <<194,57,242,156,210,120,172,248,208,88,233,168,205,61,179,61,2,85,56,96,136,83,210,119,125,103,192,129,116,221,161,132,108,92,219,163,233,228,179,34,47,164,22,72,45,203,239,245,155,67,137,138,191,142,152,135,233,57,229,36,0,69,149,197,54,204,25,147,4,223,251,99,183,6,158,247,67,220,83,177,77,117,131,196,144,151,105,108,83,227,69,84,143,238,81,189,141,167,155,224,8,199,237,102,62,92,133,120,118,133,41,142,45,9,195,211,95,49,101,226,51,225,78,198,255,142,182,113,179,20,36,214,
           62,56,147,108,248,95,132,193,0,194,102,78,45,7,102,167,183,146,79,108,190,65,108,0,92,40,164,29,81,63,204,59,252,170,33,209,139,181,226,35,136,229,147,190,3,119,238,178,83,203,7,92,103,21,22,171,75,199,87,255,171,219,146,122,207,235,188,254,247,58,64,1,31,190,233,135,232,47,71,117,82,107,254,74,23,178,186,85,110,204,185,51,113,160,41,145,190,52,63,103,210,54,235,206,82,160,188,238,209,99,90,100,138,147,84,20,30,47,253,73,245,168,13,222>>}
       ).

%%--------------------------------------------------------------------
%% Common Test interface functions -----------------------------------
%%--------------------------------------------------------------------
all() ->
    [ocsp_test].

groups() ->
    [].

%%--------------------------------------------------------------------
init_per_suite(Config) ->
    Config.

end_per_suite(_Config) ->
    ok.

%%--------------------------------------------------------------------

init_per_group(_GroupName, Config) ->
    Config.

end_per_group(_GroupName, Config) ->
    Config.

%%--------------------------------------------------------------------
init_per_testcase(_TestCase, Config) ->
    Config.

end_per_testcase(_TestCase, _Config) ->
    ok.

%%--------------------------------------------------------------------
%% Test Cases --------------------------------------------------------
%%--------------------------------------------------------------------
ocsp_test() ->
    [{doc, "Test functions in pubkey_ocsp"}].
ocsp_test(Config) when is_list(Config) ->
    undefined = pubkey_ocsp:get_nonce_extn(undefined),
    #'Extension'{extnID    = ?'id-pkix-ocsp-nonce',
                 extnValue = ?NONCE} = pubkey_ocsp:get_nonce_extn(?NONCE),

    {ok, OcspResponse = #'BasicOCSPResponse'{}} =
        pubkey_ocsp:decode_response(?OCSP_RESPONSE_DER),

    IsTrustedReponderFun = fun(_) -> true end,
    {ok, [SingleResponse], []} =
        pubkey_ocsp:verify_response(OcspResponse,
                                    [?RESPONDER_CERT],
                                    ?NONCE,
                                    ?ISSUER_CERT,
                                    IsTrustedReponderFun),
    {'SingleResponse',
     {'CertID',
      {'AlgorithmIdentifier',
       {1,3,14,3,2,26},<<5,0>>},
      <<227,147,252,182,155,101,129,45,194,162,22,93,127,46,112,193,196,28,241,232>>,
      <<34,25,129,87,115,255,155,246,200,97,92,7,51,110,152,61,97,155,164,171>>,9},
     {good,'NULL'},"20230720122949Z",asn1_NOVALUE,asn1_NOVALUE} =
        SingleResponse,

    CertId = SingleResponse#'SingleResponse'.certID,
    WrongNameHashSingleResponse =
        SingleResponse#'SingleResponse'{
          certID = CertId#'CertID'{issuerNameHash = <<"rubbish_hash">>}},
    NextUpdateFromPast = SingleResponse#'SingleResponse'{nextUpdate = "19820720122949Z"},
    ThisUpdateFromFuture = SingleResponse#'SingleResponse'{thisUpdate = "21230720122949Z"},
    {ok, SingleResponse} =
        pubkey_ocsp:find_single_response(
          ?A_SERVER_CERT, ?ISSUER_CERT,
          [rubbish_single_response,
           WrongNameHashSingleResponse,
           NextUpdateFromPast,
           ThisUpdateFromFuture,
           SingleResponse, rubbish_single_response]),

    %% invalid responses
    IsNotTrustedReponderFun = fun(_) -> false end,
    %% passing A_SERVER_CERT to get into Case3 in pubkey_oscp:is_authorized_responder
    {error,ocsp_responder_cert_not_found} =
        pubkey_ocsp:verify_response(OcspResponse,
                                    [#cert{otp = ?A_SERVER_CERT}],
                                    ?NONCE,
                                    ?ISSUER_CERT,
                                    IsNotTrustedReponderFun),
    OcspResponseWrongSignature =
        OcspResponse#'BasicOCSPResponse'{signature = <<"rubbish_signature">>},
    {error, ocsp_responder_cert_not_found} =
        pubkey_ocsp:verify_response(OcspResponseWrongSignature,
                                    [?RESPONDER_CERT],
                                    ?NONCE,
                                    ?ISSUER_CERT,
                                    IsTrustedReponderFun),

    {error, nonce_mismatch} =
        pubkey_ocsp:verify_response(OcspResponse,
                                    [?RESPONDER_CERT],
                                    <<"rubbish_nonce">>,
                                    ?ISSUER_CERT,
                                    IsTrustedReponderFun),

    OcspResponseProducedAt22ndCentury = % Year AD 2123
        OcspResponse#'BasicOCSPResponse'{
          tbsResponseData = #'ResponseData'{producedAt = "21230720122949Z"}},
    {error,ocsp_stale_response} =
        pubkey_ocsp:verify_response(OcspResponseProducedAt22ndCentury,
                                    [?RESPONDER_CERT],
                                    ?NONCE,
                                    ?ISSUER_CERT,
                                    IsTrustedReponderFun),
    ok.
