#!/bin/bash
# unplugdrive.sh
#
# Enables unmounting prior to unplugging removable storage
# Allows simultaneous selection of multiple drives
# Unmounts all mounted partitions on nominated drive(s)
# Removes mountpoints
#
# If user is expected to be able to start unplugdrive without being asked for entering his sudo password, it is necessary to add unplugdrive to the /etc/sudoers.d file with the "NOPASSWORD" option from ver. 0.82c and above.
# System settings referring to unplugdrive need to get supplemented by sudo preceding unplugdrive from ver. 0.82c and above. If etc/sudoers.d/antixers is modified at the same time, user won't notice any difference of operation compared to precvious version 0.82.
# Additionally an alias like "alias unplugdrive='sudo unplugdrive.sh'" and/or "alias unplugdrive.sh='sudo unplugdrive.sh'" should be added to system settings for user convenience, from ver 0.82c and above.
#
# Requires yad and pmount to be installed.
# Requires /etc/udev/rules.d/99-usbstorage.unused to be renamed 99-usbstorage

# bobc 12/28/19 added safemode, default FALSE, to avoid confirmation dialogs
# ppc 20/11/2020 - changed some design, making the dialog windows seem more "inbuit" with the OS, added time out to warning messages, and a title to the windows... cleaned up a bit the selection dialog so it's more multi-lingual...
# Robin.antiX 04/14/2021 new feature: debug mode structure added for easier testing.
# Robin.antiX 04/14/2021 bugfix: script did not unmount drives mounted to multiple mountpoints correctly, reporting to user it was safe to unplug while device was still mounted. First fixings by streamlining the processing.
# Robin.antiX 04/14/2021 bugfix: script did not unmount drives containing underline character "_" in mountpoint (happened always when parition-lable contained this character); fixed by removing seperator replacement and added different IFS instead.
# Robin.antiX 04/18/2021 bugfix: script did failed unmounting any usb drives not visable to df command, fixed by adding -a option.
# Robin.antiX 04/18/2021 bugfix: when more than 2 devices were to be unmounted lower checkboxes not accessible when using classical 3-button mouse. Slider at right side without function, so fixed by deriving window hight from number of drives/mountpoints displayed.
# Robin.antiX 04/18/2021 GUI behaviour: added -center to yad dialogs to prevent script windows to pop up in random, unforseeable places somewhere on screen. Moreover white borderless windows randomly scattered on white background from other open documents, resulting in text messages hiding within other text.
# Robin.antiX 04/18/2021 new features: commandline switches for help, safemode, decorations and debugging. Added taskbar icon in order to prevent yad from using its erlenmeyer flask.
# Robin.antiX 04/18/2021 bugfix: wrong entries in drivelist replaced (should have been TRUE/FALSE instead of position number for initialising checkboxes correctly)
# Robin.antiX 04/20/2021 bugfix: check for remaining devices, did still not reliably detect leftover mountpoints.
# Robin.antiX 04/20/2021 bugfix: deriving amount of time needed for sync from system info instead of fixed value; moreover before the delay window was not closing after sync due to “$!” mismatch of yad pids.
# Robin.antiX 05/04/2021 new feature: "centred" optional for users preferring opening windows near mouse pointer. Additional command line option: -c or --centred.
# Robin.antiX 05/06/2021 bugfix: replaced „pmount” by „umount” command for actual unmounting devices for several issues (pmount mixing up mountpoints for identical partitions and unfounded "Error: could not determine real path of the device" messages.)
# Robin.antiX 06/08/2021 bugfix: mechanical (rotational) USB hdds did not spun down after unmounting, so it was not safe to unplug them wheras script tells user it was. Therefore it was neccessary to change default behaviour from “unsafe” to “safe” so user has to deactivate support for spinning devices actively by passing -u or -i option.

# fixme: devices mounted to mountpoints containing blank characters still don't unmount properly.

### Please Note: text strings for debug output dont't need to get marked for translation.


# preliminaries
TEXTDOMAINDIR=/usr/share/locale
TEXTDOMAIN=unplugdrive.sh

title=$"Unplug USB device"
version="0.83"
txt_version=$"Version "$version

icon_taskbar="/usr/share/icons/numix-square-antix/48x48/devices/usbpendrive_unmount.png"
declare -a mountpointlist
#TEMPFILE=/tmp/unpluglist    # no longer needed
#debuglog="$HOME/unplugdrive-debug-$(date +%d%m%Y%H%M%S).log"	# function to write a logfile not implemented yet
removablelist=""
removablenow=""
summarylist=""
drivenow=""
driveprevious=""
drivelist=""
deviceslist=()
spindownlist=""
spinningdownlist=""
txt_rotationalerror=""
umountfailurelist=()
flag_debugmode=0
flag_dryrun=0
flag_spindown=0
flag_umounterror=0
flag_ignore_rotational=0
decorations="--undecorated"
centred=""
flag_safemode="TRUE"
CHR_0x0A="$(printf '\nx')"; CHR_0x0A=${CHR_0x0A%x}		# create special seperator
txt_on=$"on" 											# refers to message “<partition> on <mountpoint>” presented in dialog to make it translatable.
true_user=$SUDO_USER


# check for commandline options
txt_cline="$*"
while [[ $# -gt 0 ]]
do
  opt_cline="$1"
  case $opt_cline in
    -h|--help)	      echo ""
                      echo $"  Unplugdrive" "($txt_version)"
                      echo ""
                      echo -e $"  GUI tool for safely unplugging removable USB devices."
                      echo ""
                      echo $"  Usage: unplugdrive.sh [options]"
                      echo ""
                      echo $"  Options:"
                      echo -e $"\t-h or --help\t\tdisplays this help text"
                      echo -e $"\t-u or --unsafe\t\truns script in unsafe mode,\n\t            \t\tomiting some checks and dialogs"
                      echo -e $"\t-i or --ignore\t\tignores whether devices are\n\t            \t\treported as rotational even in\n\t            \t\tdefault mode. No spindown."
                      echo -e $"\t-d or --decorated\tuses window decorations"
                      echo -e $"\t-c or --centred\t\topen dialogs centred"
                      echo -e $"\t-g or --debug\t\tenable debug output (terminal)"
                      echo -e $"\t-p or --pretend\t\tdry run, don't actually un-\n\t            \t\tmount drives (for debugging)"
                      echo -e ""
                      echo -e $"  NEVER use the options -u and -i on rotational devices!"
                      echo ""
                      echo -e $"  Questions, Suggestions and Bugreporting please to:"
                      echo -e "\t<forum.antiXlinux.com>"
                      echo -e ""
                      echo -e "  Copyright 2011, 2012-2018 SamK, anticapitalista"		# entry needed to comply with EU legislation
                      echo -e "  Copyright 2019, 2020, 2021 the antiX comunity"			# entry needed to comply with EU legislation
                      echo -e ""										# This GPL text may not get translated. See gnu.org for details.
                      echo -e "  This program is free software: you can redistribute it and/or"
                      echo -e "  modify it under the terms of the GNU General Public License as"
                      echo -e "  published by the Free Software Foundation, either version 3 of"
                      echo -e "  the License, or (at your option) any later version."
                      echo ""
                      echo -e "  This program is distributed in the hope that it will be useful,"
                      echo -e "  but WITHOUT ANY WARRANTY; without even the implied warranty of"
                      echo -e "  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the"
                      echo -e "  GNU General Public License for more details."
                      echo ""
                      echo -e "  You should have received a copy of the GNU General Public"
                      echo -e "  License along with this program.  If not, see "
                      echo -e "  \t<http://www.gnu.org/licenses/>."
                      echo ""
                      exit 0;;
    -d|--decorated)   decorations=""
                      shift;;
    -c|--centred)     centred="--center"
                      shift;;                      
    -u|--unsafe)      flag_safemode="FLASE"
                      shift;;
    -i|--ignore)      flag_ignore_rotational=1
                      shift;;
    -g|--debug)       flag_debugmode=1
                      shift;;
    -p|--pretend)     flag_dryrun=1
                      shift;;                      
	 *)               echo -e $"Invalid command line argument. Please call\nunplugdrive -h or --help for usage instructions."
					  exit 1
                      shift;;
  esac
done

[ $flag_debugmode == 1 ] && echo -e "Version:\t$version\nDebug mode: on""\n$txt_cline\nsafemode:\t$flag_safemode\ndecorations:\t$decorations\ncentred:\t$centred\ndryrun (pretend):\t$flag_dryrun\nignore rotationals\t$flag_ignore_rotational\nTrue user:\t$true_user\nexecuted by:\t`whoami`"

# check whether unplugdrive is run with sudo by normal user.
if [ `whoami` != "root" ]; then
	if [ "$true_user" != "root" ]; then echo "“unplugdrive” should be run with SUDO." && exit 1; else echo "“unplugdrive” should not be run by root, please use SUDO being normal user." && exit 1; fi
else
	if [ "$true_user" == "root" ] || [ "$true_user" == "" ]; then echo "“unplugdrive” should not be run by root, please use SUDO being normal user." && exit 1; fi
fi

#check for needed helper programs etc.:
[ ! $(command -v yad) ] && echo $"\nˮyadˮ is not installed.\n   --> Please install ˮyadˮ before executing this script.\n" && exit 1
[ -e icon_taskbar ] && echo $"Taskbar icon $icon_taskbar not found."


# preparing cleanup on leaving
function cleanup {
[ -e $TEMPFILE ] && rm -f $TEMPFILE
# echo "mischief managed!"
return 0
}
trap 'cleanup && exit 1' SIGINT

# prepare safemode message
function safemode {
if [ "$flag_safemode" = "TRUE" ]; then
  sudo -u $true_user -- yad --title="$title" --fixed $centred --timeout=3 --mouse $decorations \
      --window-icon="$icon_taskbar" \
      --text=$"<b>Aborting</b> on user request\n<b>without unmounting.</b>" \
      --no-buttons
  buttonselect=$?
  [ $buttonselect == 252 ] || [ $buttonselect == 255 ] && buttonselect=1
  [ $buttonselect == 70 ] && buttonselect=1
fi
return 0
}

# 1.) Preparations for user dialog:
# ---------------------------------

# Collect details of each removable device that has at least one mounted partition.
# This is the base list for processing
discovered=$(sudo -u $true_user -- pmount|grep /dev/|sort|cut -d' ' -f 1,2,3)

# Create a raw list of all removable devices excluding CD/DVD, containing each removable drive, mounted partition and mountpoint.
orig_IFS=$IFS															# store original field seperator
IFS="$CHR_0x0A"															# modify field seperator
[ $flag_debugmode == 1 ] && echo "1.1.) raw list (\$discovered): "
for item in $discovered;do
   [ $flag_debugmode == 1 ] && echo -e "removable device discovered:\t$item"
   if [[ ! $item = /dev/sr* ]];then
     detectedlist=$(echo -e "$detectedlist\n$item")
   fi
done

# Derive a tailored "removablelist" from first "detectedlist", stripped of all unneccessary string elements. Only names of devices and mountpoints are left.
# This list is informational only, for display of mounted partitions in yad dialog!
[ $flag_debugmode == 1 ] && echo "1.2.) tailored list for display purposes (\$removablenow):"
for item in $detectedlist;do
		removablenow="$(echo $item | cut -c 6- | sed 's:/.*/:/:' | sed 's/[/]//g' | sed 's/ on /\t'$txt_on'\t/g')"
		[ $flag_debugmode == 1 ] && echo -e "removablenow item:\t$removablenow"
   removablelist="$removablelist$removablenow\n"
done

# Derive another tailored "drivelist" from first "detectedlist", stripped of all string elements except for devicenames. Count mountpoints and unique devices.
# This list is functional, devicename entries are used later for looking up of devices marked by user in yad dialog for unmount.
[ $flag_debugmode == 1 ] && echo "1.3.) second tailored list for lookup of devices (\$drivelist):"
position=0;drnumber=0
for item in $detectedlist;do
   drivenow=$(echo $item|cut -d' ' -f 1|cut -c 6-8)
   [ $flag_debugmode == 1 ] && echo -e "drivelist item:\t$drivenow"
   if [ "$drivenow" != "$driveprevious" ];then
      drivelist="$drivelist FALSE $drivenow"							# "FALSE" here lets checkboxes unchecked, (TRUE) will check them. Entry needed for yad cheklist.
      driveprevious=$drivenow
      let $((drnumber++))
   fi
   let $((position++))													# found vacant $position entry now is reused (together with $drnumber) for estimate necessary height of yad window.
done
[ $flag_debugmode == 1 ] && echo -e "\$drivelist completed (unique items):\t$drivelist\nNumber of partitions (\$position):\t$position\nNumber of devices (\$drnumber):\t$drnumber"
IFS=${orig_IFS}															# reset field seperator to original values


# Display a message that no candidate for unmounting was discovered
if [ -z "$drivelist" ];then
   sudo -u $true_user -- yad --title="$title" --fixed $centred --timeout=5 --mouse $decorations \
       --window-icon="$icon_taskbar" \
       --text $"A removable drive with a mounted\npartition was not found.\n<b>It is safe to unplug the drive(s)</b>" \
       --no-buttons 
   exit 0
fi


# 2.) User dialog for selection
# -----------------------------

# User Dialog, display a list from which the drives to be unplugged may be selected.
while [ -z "$selected" ]  # Don't proceed as long nothing was selected, or abort.
do
	height=$((22*$position+25*$drnumber+50))
	[ $flag_debugmode == 1 ] && echo -e "2.) User dialog (Yad):\nheight:\t$height"
	selected=$(sudo -u $true_user -- yad --title="$title" --fixed $centred --height="$height" --width="300" --mouse $decorations \
				   --window-icon="$icon_taskbar" \
				   --text=$"<b>Mounted USB partitions:</b>\n$removablelist<b>Choose the drive(s) to be unplugged:</b>" \
				   --list --separator=" " --no-headers --no-selection \
				   --checklist --column=" " --column="Drive(s)"  $drivelist)
	buttonselect=$?
	[ $buttonselect == 252 ] || [ $buttonselect == 255 ] && buttonselect=1		# Catch ESC button and X-icon also 
	[ $buttonselect == 1 ] && break
done

[ -z "$selected" ] && safemode
[ $buttonselect == 70 ] || [ $buttonselect == 1 ] && cleanup && exit 1										# exit for both, normal and safe mode.
[ $flag_debugmode == 1 ] && echo -en "yad output “\$selected”:\n$selected\n"


# 3.) Preparations for unmounting
#--------------------------------

# Create two new lists of mountpoints used by the drives selected to be unplugged, one for unmounting, the other for display purpose.
# (remark: wouldn't this better have been retrieved from base list above?)
[ $flag_debugmode == 1 ] && echo "3.1.) Lists for unmounting (\$mountpointlist and \$summarylist)"
v=0; u=0;
for item in $selected;do
	[ $flag_debugmode == 1 ] && echo -e "selected item (\$selected):\t$item"
	if [[ "$item" != "TRUE" && "$item" != "FALSE" ]]; then
		v=${#mountpointlist[@]}
		mapfile -d"$CHR_0x0A" -O $v mountpointlist <<< "$(sudo -u $true_user -- df -a |grep $item | tr -s ' ' | cut -d' ' -f 6)"	# montpointlist for unmounting devices actually
		summarylist="$summarylist$(sudo -u $true_user -- df -a --output=source,target |grep $item)\n"								# summarylist for display in yad
		deviceslist+=("/dev/$item")																			# devices to check for presence after unmount
    fi
done
if [ $flag_debugmode == 1 ]; then
	while [ $u -lt ${#mountpointlist[@]} ]
		do
			echo -en "item $u in mountpointlist:\t${mountpointlist[$((u++))]}"
		done
	echo -e "Number of items to get unmounted:\t${#mountpointlist[@]}"
	echo -e "summary list is (\$summarylist):\n$summarylist"
	echo -e "Unique devices to check for after unmount (\$deviceslist):\t${deviceslist[*]}"
fi


# Obtain confirmation to proceed with unmounting (only in safemode)
if [ "$flag_safemode" = "TRUE" ]; then
	sudo -u $true_user -- yad --title="$title" --fixed $centred --mouse $decorations \
		--window-icon="$icon_taskbar" \
		--text=$"<b>About to unmount:</b>\n$summarylist<b>Please confirm you wish to proceed.</b>"
	buttonselect=$?
	[ $buttonselect == 252 ] || [ $buttonselect == 255 ] && buttonselect=1		# Catch ESC button and X-icon also 
	[ $buttonselect == 70 ] || [ $buttonselect == 1 ] && cleanup && safemode && exit 1

# Ensure user waits long enough before unplugging so everything gets written to storage (only in safemode)
	# get yad pid of specific yad window ($! will fail on yad windows)
	yad_pid_before="$(pidof -x yad) "				# blank suffix intentionally
	sudo -u $true_user -- yad --title="$title" --fixed $centred --mouse $decorations \
        --text=$"Data is being written\nto devices. <b>Please wait...</b>" \
        --window-icon="$icon_taskbar" \
        --no-buttons &
	buttonselect=$?
	[ $buttonselect == 252 ] || [ $buttonselect == 255 ] && buttonselect=1		# Catch ESC button and X-icon also 
	[ $buttonselect == 70 ] || [ $buttonselect == 1 ] && cleanup && safemode && exit 1
	sleep 1											# wait for second yad pid
	yad_pid="$(sudo -u $true_user -- pidof -x yad)"
	if [ "$yad_pid_before" != " " ]; then			# if no items sed will fail
		while read -d' ' item
		do
			yad_pid="$(echo "$yad_pid" | sed "s/$item//g")"
		done <<<"$yad_pid_before"
	fi
fi

# Write data to device(es)
sudo -u $true_user -- sync

# wait and close delay window only after sync is done (only in safe mode)
if [ "$flag_safemode" = "TRUE" ]; then
	i=0
	while :;
	do
		if [ "$(cat /proc/meminfo | grep 'Dirty:' | tr -s ' ' | cut -d' ' -f 2)" == "0" ] && [ "$(cat /proc/meminfo | grep 'Writeback:' | tr -s ' ' | cut -d' ' -f 2)" == "0" ]; then
			if [ $flag_debugmode == 1 ]; then echo "sync done"; fi; break
		fi
		sleep .2				# wait until sync is done before next check.
		i=$[$i+1]
		if [ $i -ge 5 ]; then
			i=0
			sudo -u $true_user -- sync		#feed some additional sync commands in case first check was missed on systems under heavy load.
		fi
	done
	[ $flag_debugmode == 1 ] && echo -e "pids of existing yad processes (\$yad_pid_before):\n$yad_pid_before\npid of yad info window (\$yad_pid):\n$yad_pid"
	kill $yad_pid
fi


# 4.) Unmounting selected devices
# -------------------------------

# unmount each item in $mountpointlist
[ $flag_debugmode == 1 ] && echo -e "4.) Unmounting\nNumber of items in \$mountpointlist():  ${#mountpointlist[@]}"
[ $flag_dryrun == 1 ] && echo -e "\033[1;33mDry run. Nothing was actually unmounted.\nOmit option -p for real operation.\033[0m"
i=0
while [ $i -lt ${#mountpointlist[@]} ]
do
	[ $flag_debugmode == 1 ] && echo "unmounting item $i is ${mountpointlist[i]}"
	[ $flag_dryrun != 1 ] && umount "$(echo ${mountpointlist[$i]})"						# here selected devices get unmounted actually. (it needs 'echo' for some reason)
	i=$[$i+1]
done



# 5.) Test for remaining mounted partitions, check for rotational devices 
# -----------------------------------------------------------------------

# check $deviceslist against all currtently mounted devices using findmnt command to get reliable data. Refering to “man mount”: >>For more robust and customizable output use findmnt(8), especially in your scripts.<<
[ $flag_debugmode == 1 ] && echo -e "5.) Check for remaining unmounted partitions\nand check whether unmounted devices need spindown."
u=0
while [ $u -lt ${#deviceslist[@]} ]
	do
		checkitem="$(sudo -u $true_user -- findmnt --list -o SOURCE,TARGET | grep ${deviceslist[u]})"
		[ $flag_debugmode == 1 ] &&  echo -e "checking for item $u in \$deviceslist[n]:\n${deviceslist[u]}"
		if [ -n "$checkitem" ]; then
			umountfailurelist+=("${deviceslist[u]}")
			echo -e $"\033[1;31mWARNING: DEVICE ${deviceslist[u]} WAS NOT PROPERLY UNMOUNTED!\033[0m\n\033[4;37mPlease check before unplugging.\033[0m"
			flag_umounterror=1
			v=0
			while [ $v -lt ${#mountpointlist[@]} ]
				do
					mountpointerrorlist="$mountpointerrorlist$(echo ${mountpointlist[v]} | grep $(echo ${deviceslist[u]} | cut -c 6-))\n"
					let $((v++))
				done
		else
			[ $flag_debugmode == 1 ] &&  echo -e "Check OK"
			# check which devices are reported as rotational and need to spin down.
			if [ $(cat /sys/block/$(echo ${deviceslist[u]} | cut -d '/' -f 3)/queue/rotational) == 1 ]; then
				[ $flag_debugmode == 1 ] &&  echo -e "Device $(echo ${deviceslist[u]} | cut -d '/' -f 3) is reported as “rotational”. Needs spindown."
				spindownlist="$spindownlist${deviceslist[u]} "  # hint: since devcices in /dev never contain blanks it is safe here to use it as separator.
				flag_spindown=1
			fi
		fi
		let $((u++))
    done


# 6.) Spin down rotational devices and wait for them reporting to be spun down.
# -----------------------------------------------------------------------------

# spin down device if it is marked as rotating. Otherwise it is still NOT safe to unplug. (in safemode only)
if [ $flag_ignore_rotational == 1 ]; then flag_spindown=0; [ $flag_debugmode == 1 ] && echo -e "Ignoring rotational devices in need for spindown per user request"; fi
if [ "$flag_safemode" = "TRUE" ] && [ $flag_spindown == 1 ]; then
		echo -e $"Please wait until device(s) spun down..."
		# get yad pid of specific yad window ($! will fail on yad windows)
		yad_pid_before="$(pidof -x yad) "			# blank suffix intentionally
		sudo -u $true_user -- yad --title="$title" --fixed $centred --mouse $decorations \
			--text=$"Rotational devices\nspin down. <b>Please wait...</b>" \
			--window-icon="$icon_taskbar" \
			--no-buttons &
		sleep 1										# wait for second yad pid
		yad_pid="$(pidof -x yad)"
		if [ "$yad_pid_before" != " " ]; then		# if no items sed will fail
			while read -d' ' item
			do
				yad_pid="$(echo "$yad_pid" | sed "s/$item//g")"
			done <<<"$yad_pid_before"
		fi
		[ $flag_debugmode == 1 ] && echo -e "6.1) Spinning down rotationals\nItems in spindownlist: $(echo "$spindownlist" | wc -w)"
		while read -d ' ' item; do
			hdparm -Y "$item"					# we can safely spin down rotational devices even if still mounted (can only happen in dryrun mode), they will power up again when accessed.
			response=$?
			if [ $response == 0 ];then
				spinningdown=1
				spinningdownlist="$spinningdownlist$item "
			else
				[ $flag_debugmode == 1 ] && echo -e "Item does not respond correctly to spindown command: $item\nItem removed from list."
				txt_rotationalerror="$txt_rotationalerror$item "
			fi
		done <<<"$spindownlist"
		[ "$(echo "$spinningdownlist" | wc -w)" == 0 ] && flag_spindown=0
		[ $flag_debugmode == 1 ] && echo -e "6.2) Waiting for devices to report spun down\nItems in spinningdownlist: $(echo "$spinningdownlist" | wc -w)\nspinningdownlist: $spinningdownlist\nflag_spindown: $flag_spindown"
# Following lines deactivated due to incompatibility to some drives, which don't respond to status query once received spindown command until they get accessed again and are spun up.
# So we'll have to rely on exit status of first spindown command only for now.
#		# wait and check until drives are reported as spun down (only if device responds correctly)
#		while [ $flag_spindown == 1 ]; do
#			flag_spindown=0
#			while read -d ' ' item; do
#				[ $flag_debugmode == 1 ] && echo --- $item	---
#				[ -e "$(hdparm -C $item | grep -o standby)" ] && flag_spindown=1
#				[ $flag_debugmode == 1 ] && echo -e "item: $item \t $flag_spindown"
#			done <<<"$spinningdownlist"
#			sleep 1  # give drives some time to report
#			#[ -e "$(ps | grep '$(echo $yad_pid | cut -d' ' -f 1) \| $(echo $yad_pid | cut -d' ' -f 2)')" ] && flag_spindown=0 # go on if a device doesn't report correctly and user is sure all devices are spun down.
#		done
		[ $flag_debugmode == 1 ] && echo -e "Spindown for all requested devices done.\npids of existing yad processes (\$yad_pid_before):\n$yad_pid_before\npid of yad info window (\$yad_pid):\n$yad_pid"
		kill $yad_pid
fi		


# 7.) Final user dialog on exit
# -----------------------------

if [ "$txt_rotationalerror" != "" ]; then
	successtext=$"<b>Unmounted:</b>\n$summarylist\nBut despite the following devices were reported to be rotational\nthey did not respond to spindown command:\n\t$txt_rotationalerror\nPlease check whether they are spun down before unplugging.\n<b>After rotational drives are spun down it is safe to unplug.</b>"
	yad_timeout=""
	yad_button="--button="OK""
else
	successtext=$"<b>Unmounted:</b>\n$summarylist\n<b>It is safe to unplug the drive(s)</b>"
	yad_timeout="--timeout=5"
	yad_button="--no-buttons"
fi

# User information on exit:
if [ $flag_umounterror == 1 ];then
	# Display a message if unmount failed
	mountpointerrorlist="$(echo -e $mountpointerrorlist |tr -s '\n')"
	sudo -u $true_user -- yad --title="$title" --fixed $centred --mouse $decorations \
	    --text=$"<b>Mountpoint removal failed.</b>\n<u>One or more mountpoin(s) remain present at:</u>""\n$mountpointerrorlist\n"$"<b>Check each mountpoint listed before unpluging the drive(s).</b>" \
        --window-icon="$icon_taskbar" \
        --button="OK"
		# no need for checking button selection anymore
	cleanup && exit 1
else
   # Display a message if unmount successful   
	sudo -u $true_user -- yad --title="$title" --fixed $centred $yad_timeout --mouse $decorations \
        --text="$successtext" \
        --window-icon="$icon_taskbar" \
        $yad_button
        # no need for checking button selection anymore
	cleanup && exit 0
fi


