"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerParaRoute = registerParaRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _saved_objects_paragraphs_router = require("../../adaptors/notebooks/saved_objects_paragraphs_router");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerParaRoute(router) {
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphIndex: _configSchema.schema.number(),
        paragraphInput: _configSchema.schema.string(),
        inputType: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const saveResponse = await (0, _saved_objects_paragraphs_router.createParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph/clearall`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const clearParaResponse = await (0, _saved_objects_paragraphs_router.clearParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: clearParaResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      query: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const params = {
      noteId: request.query.noteId,
      paragraphId: request.query.paragraphId
    };
    try {
      const deleteResponse = await (0, _saved_objects_paragraphs_router.deleteParagraphs)(params, context.core.savedObjects.client);
      return response.ok({
        body: deleteResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph/update/run`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string(),
        paragraphType: _configSchema.schema.string(),
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        dataSourceMDSLabel: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      const runResponse = await (0, _saved_objects_paragraphs_router.updateRunFetchParagraph)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: runResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        paragraphInput: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const saveResponse = await (0, _saved_objects_paragraphs_router.updateFetchParagraph)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/set_paragraphs`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          output: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
            unknowns: 'allow'
          }))),
          input: _configSchema.schema.object({
            inputText: _configSchema.schema.string(),
            inputType: _configSchema.schema.string()
          }),
          dateCreated: _configSchema.schema.string(),
          dateModified: _configSchema.schema.string(),
          id: _configSchema.schema.string(),
          dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
            defaultValue: ''
          })),
          dataSourceMDSLabel: _configSchema.schema.maybe(_configSchema.schema.string({
            defaultValue: ''
          }))
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const updateNotebook = {
        paragraphs: request.body.paragraphs,
        dateModified: new Date().toISOString()
      };
      const updateResponse = await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId, {
        savedNotebook: updateNotebook
      });
      return response.ok({
        body: updateResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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