//
// tardy - a tar post-processor
// Copyright (C) 2011 Peter Miller
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or (at
// your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
//

#ifndef LIBTARDY_FILE_INPUT_GUNZIP_H
#define LIBTARDY_FILE_INPUT_GUNZIP_H

#include <libtardy/ac/zlib.h>

#include <libtardy/file/input.h>

/**
  * The file_input_gunzip class is used to represent an input stream which is
  * uncompressed on the fly.
  */
class file_input_gunzip:
    public file_input
{
public:
    /**
      * The destructor.
      */
    virtual ~file_input_gunzip();

    /**
      * The create class method is used to create new dynamically
      * allocated instance of this class.
      *
      * @param deeper
      *     The deeper input which this filter reads from.
      */
    static pointer create(const file_input::pointer &deeper);

    /**
      * The open class method is used to sniff at the input file, to see
      * if it is compressed.  If it is compressed, the #create class
      * method will be called, otherwise the deeper file will be returned
      * unfiltered.
      *
      * @param deeper
      *     The deeper input which is to be filtered (conditionally).
      */
    static pointer open(const file_input::pointer &deeper);

protected:
    // See base class for documentation.
    rcstring filename(void) const;

    // See base class for documentation.
    int read_inner(void *data, size_t data_size);

    // See base class for documentation.
    unsigned long get_position_inner(void) const;

    /**
      * The candidate class method is used to check the magi number of a
      * gzipped file.  All of the bytes read are unread before this method
      * returns.
      */
    static bool candidate(const file_input::pointer &deeper);

private:
    /**
      * The constructor.
      * It is private on purpose, use the #create class method instead.
      *
      * @param deeper
      *     The deeper input which this filter reads from.
      */
    file_input_gunzip(const file_input::pointer &deeper);

    /**
      * The deeper instance variable is used to remember the deeper
      * input which this filter reads from.
      */
    file_input::pointer deeper;

    z_stream stream;
    bool z_eof;
    uLong crc;
    unsigned long pos;
    Byte *buf;

    void zlib_fatal_error(int err);

    /**
      * The get_long method is used to extract a 4-byte little-endian
      * value from the deeper input stream.
      */
    unsigned long get_long(void);

    /**
      * The get_short method is used to extract a 2-byte little-endian
      * value from the deeper input stream.
      */
    unsigned get_short(void);

    /**
      * The get_header method is used to read and check the GZIP file
      * header in the input stream.
      */
    void read_header(void);

    /**
      * The skip_string method is sued to advance the deeper file
      * position past a NUL-terminated string in the deeper input.
      */
    void skip_string(void);

    /**
      * The default constructor.  Do not use.
      */
    file_input_gunzip();

    /**
      * The copy constructor.  Do not use.
      */
    file_input_gunzip(const file_input_gunzip &arg);

    /**
      * The assignment operator.  Do not use.
      */
    file_input_gunzip &operator=(const file_input_gunzip &arg);
};

#endif // LIBTARDY_FILE_INPUT_GUNZIP_H
