/***************************************************************************//**
 *   @file   stm32/stm32_spi.c
 *   @brief  Implementation of stm32 spi driver.
 *   @author Darius Berghe (darius.berghe@analog.com)
********************************************************************************
 * Copyright 2020(c) Analog Devices, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. Neither the name of Analog Devices, Inc. nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES, INC. “AS IS” AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL ANALOG DEVICES, INC. BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/
#include <stdlib.h>
#include <errno.h>
#include "no_os_util.h"
#include "no_os_gpio.h"
#include "stm32_gpio.h"
#include "no_os_spi.h"
#include "stm32_spi.h"
#include "no_os_delay.h"
#include "no_os_alloc.h"
#include "stm32_dma.h"
#ifdef HAL_TIM_MODULE_ENABLED
#include "no_os_pwm.h"
#endif

static int stm32_spi_config(struct no_os_spi_desc *desc)
{
	int ret;
	uint32_t prescaler = 0u;
	SPI_TypeDef *base = NULL;
	struct stm32_spi_desc *sdesc = desc->extra;

	/* automatically select prescaler based on max_speed_hz */
	if (desc->max_speed_hz != 0u) {
		uint32_t div = sdesc->input_clock / desc->max_speed_hz;

		switch (div) {
		case 0 ... 2:
			prescaler = SPI_BAUDRATEPRESCALER_2;
			break;
		case 3 ... 4:
			prescaler = SPI_BAUDRATEPRESCALER_4;
			break;
		case 5 ... 8:
			prescaler = SPI_BAUDRATEPRESCALER_8;
			break;
		case 9 ... 16:
			prescaler = SPI_BAUDRATEPRESCALER_16;
			break;
		case 17 ... 32:
			prescaler = SPI_BAUDRATEPRESCALER_32;
			break;
		case 33 ... 64:
			prescaler = SPI_BAUDRATEPRESCALER_64;
			break;
		case 65 ... 128:
			prescaler = SPI_BAUDRATEPRESCALER_128;
			break;
		default:
			prescaler = SPI_BAUDRATEPRESCALER_256;
			break;
		}
	} else
		prescaler = SPI_BAUDRATEPRESCALER_64;

	switch (desc->device_id) {
#if defined(SPI1)
	case 1:
		base = SPI1;
		break;
#endif
#if defined(SPI2)
	case 2:
		base = SPI2;
		break;
#endif
#if defined(SPI3)
	case 3:
		base = SPI3;
		break;
#endif
		break;
#if defined(SPI4)
	case 4:
		base = SPI4;
		break;
#endif
#if defined(SPI5)
	case 5:
		base = SPI5;
		break;
#endif
#if defined(SPI6)
	case 6:
		base = SPI6;
		break;
#endif
	default:
		ret = -EINVAL;
		goto error;
	};
	sdesc->hspi.Instance = base;
	sdesc->hspi.Init.Mode = SPI_MODE_MASTER;
	sdesc->hspi.Init.Direction = SPI_DIRECTION_2LINES;
	sdesc->hspi.Init.DataSize = SPI_DATASIZE_8BIT;
	sdesc->hspi.Init.CLKPolarity = desc->mode & NO_OS_SPI_CPOL ?
				       SPI_POLARITY_HIGH :
				       SPI_POLARITY_LOW;
	sdesc->hspi.Init.CLKPhase = desc->mode & NO_OS_SPI_CPHA ? SPI_PHASE_2EDGE :
				    SPI_PHASE_1EDGE;
	sdesc->hspi.Init.NSS = SPI_NSS_SOFT;
	sdesc->hspi.Init.BaudRatePrescaler = prescaler;
	sdesc->hspi.Init.FirstBit = desc->bit_order ? SPI_FIRSTBIT_LSB :
				    SPI_FIRSTBIT_MSB;
	sdesc->hspi.Init.TIMode = SPI_TIMODE_DISABLE;
	sdesc->hspi.Init.CRCCalculation = SPI_CRCCALCULATION_DISABLE;
	sdesc->hspi.Init.CRCPolynomial = 10;
	ret = HAL_SPI_Init(&sdesc->hspi);
	if (ret != HAL_OK) {
		ret = -EIO;
		goto error;
	}
#ifdef SPI_SR_TXE
	__HAL_SPI_ENABLE(&sdesc->hspi);
#endif
	return 0;
error:
	return ret;
}

/**
 * @brief Initialize the SPI communication peripheral.
 * @param desc - The SPI descriptor.
 * @param param - The structure that contains the SPI parameters.
 * @return 0 in case of success, -1 otherwise.
 */
int32_t stm32_spi_init(struct no_os_spi_desc **desc,
		       const struct no_os_spi_init_param *param)
{
	int32_t ret;
	struct no_os_spi_desc	*spi_desc;

	if (!desc || !param)
		return -EINVAL;

	spi_desc = (struct no_os_spi_desc *)no_os_calloc(1, sizeof(*spi_desc));
	if (!spi_desc)
		return -ENOMEM;

	struct stm32_spi_desc *sdesc;
	struct stm32_spi_init_param *sinit;

	sdesc = (struct stm32_spi_desc*)no_os_calloc(1, sizeof(struct stm32_spi_desc));
	if (!sdesc) {
		ret = -ENOMEM;
		goto error;
	}

	spi_desc->extra = sdesc;
	sinit = param->extra;

	/* copy settings to device descriptor */
	spi_desc->device_id = param->device_id;
	spi_desc->max_speed_hz = param->max_speed_hz;
	spi_desc->mode = param->mode;
	spi_desc->bit_order = param->bit_order;
	spi_desc->chip_select = param->chip_select;
	if (sinit->get_input_clock)
		sdesc->input_clock = sinit->get_input_clock();

	ret = stm32_spi_config(spi_desc);
	if (ret)
		goto error;

	if (sinit->dma_init) {
		ret = no_os_dma_init(&sdesc->dma_desc, sinit->dma_init);
		if (ret)
			goto error;

		if (sinit->rxdma_ch) {
			sdesc->rxdma_ch = &sdesc->dma_desc->channels[0];
			sdesc->rxdma_ch->id = sinit->rxdma_ch->hdma;
			sdesc->rxdma_ch->extra = sinit->rxdma_ch;
			sdesc->rxdma_ch->irq_num = sinit->irq_num;
		}
		if (sinit->txdma_ch) {
			sdesc->txdma_ch = &sdesc->dma_desc->channels[1];
			sdesc->txdma_ch->id = sinit->txdma_ch->hdma;
			sdesc->txdma_ch->extra = sinit->txdma_ch;
		}
	}

#ifdef HAL_TIM_MODULE_ENABLED
	if (sinit->pwm_init) {
		ret = no_os_pwm_init(&sdesc->pwm_desc, sinit->pwm_init);
		if (ret)
			goto error;

		ret = no_os_pwm_disable(sdesc->pwm_desc);
		if (ret)
			goto error_pwm;
	}
	if (sinit->tx_pwm_init) {
		ret = no_os_pwm_init(&sdesc->tx_pwm_desc, sinit->tx_pwm_init);
		if (ret)
			goto error;

		ret = no_os_pwm_disable(sdesc->tx_pwm_desc);
		if (ret)
			goto error_pwm;
	}
#endif
	sdesc->csip_extra.mode = GPIO_MODE_OUTPUT_PP;
	sdesc->csip_extra.speed = GPIO_SPEED_FREQ_LOW;
	sdesc->csip.port = sinit->chip_select_port;
	sdesc->csip.number = param->chip_select;
	sdesc->csip.pull = NO_OS_PULL_NONE;
	sdesc->csip.extra = &sdesc->csip_extra;
	sdesc->csip.platform_ops = &stm32_gpio_ops;
	sdesc->alternate =  sinit->alternate;
	ret = no_os_gpio_get(&sdesc->chip_select, &sdesc->csip);
	if (ret)
		goto error;

	ret = no_os_gpio_direction_output(sdesc->chip_select, NO_OS_GPIO_HIGH);
	if (ret)
		goto error;
	*desc = spi_desc;

	return 0;

error_pwm:
#ifdef HAL_TIM_MODULE_ENABLED
	no_os_pwm_remove(sdesc->pwm_desc);
	no_os_pwm_remove(sdesc->tx_pwm_desc);
#endif
error:
	no_os_dma_remove(sdesc->dma_desc);
	no_os_free(spi_desc);
	no_os_free(sdesc);
	return ret;
}

/**
 * @brief Free the resources allocated by no_os_spi_init().
 * @param desc - The SPI descriptor.
 * @return 0 in case of success, -1 otherwise.
 */
int32_t stm32_spi_remove(struct no_os_spi_desc *desc)
{
	struct stm32_spi_desc *sdesc;

	if (!desc || !desc->extra)
		return -EINVAL;

	sdesc = desc->extra;
#ifdef HAL_TIM_MODULE_ENABLED
	no_os_pwm_remove(sdesc->pwm_desc);
	no_os_pwm_remove(sdesc->tx_pwm_desc);
#endif

	no_os_dma_remove(sdesc->dma_desc);

#ifdef SPI_SR_TXE
	__HAL_SPI_DISABLE(&sdesc->hspi);
#endif
	HAL_SPI_DeInit(&sdesc->hspi);
	no_os_gpio_remove(sdesc->chip_select);
	no_os_free(desc->extra);
	no_os_free(desc);
	return 0;
}

/**
 * @brief enable CS gpio alternate function
 * @param desc - The SPI descriptor
 * @param enable - enable = alternate function active, disable = gpio mode
 * @return 0 in case of success, errno codes otherwise.
 */
int32_t stm32_spi_altrnate_cs_enable(struct no_os_spi_desc *desc, bool enable)
{
	struct stm32_spi_desc *sdesc = desc->extra;
	int ret;

	no_os_gpio_remove(sdesc->chip_select);
	sdesc->chip_select = NULL;

	if (enable) {
		sdesc->csip_extra.mode = GPIO_MODE_AF_PP;
		sdesc->csip_extra.alternate = sdesc->alternate;
	} else {
		sdesc->csip_extra.mode = GPIO_MODE_OUTPUT_PP;
	}

	ret = no_os_gpio_get(&sdesc->chip_select, &sdesc->csip);
	if (ret)
		return ret;

	ret = no_os_gpio_direction_output(sdesc->chip_select, NO_OS_GPIO_HIGH);
	if (ret) {
		no_os_gpio_remove(sdesc->chip_select);
		return ret;
	}

	return 0;
}

/**
 * @brief Write/read multiple messages to/from SPI.
 * @param desc - The SPI descriptor.
 * @param msgs - The messages array.
 * @param len - Number of messages.
 * @return 0 in case of success, errno codes otherwise.
 */
int32_t stm32_spi_transfer(struct no_os_spi_desc *desc,
			   struct no_os_spi_msg *msgs,
			   uint32_t len)
{
	struct stm32_spi_desc *sdesc;
	struct stm32_gpio_desc *gdesc;
	uint64_t slave_id;
	static uint64_t last_slave_id;
	int ret = 0;

	if (!desc || !desc->extra || !msgs)
		return -EINVAL;

	sdesc = desc->extra;
	gdesc = sdesc->chip_select->extra;
#ifdef SPI_SR_TXE
	uint32_t tx_cnt = 0;
	uint32_t rx_cnt = 0;
	SPI_TypeDef * SPIx = sdesc->hspi.Instance;
#endif

	// Compute a slave ID based on SPI instance and chip select.
	// If it did not change since last call to stm32_spi_write_and_read,
	// no need to reconfigure SPI. Otherwise, reconfigure it.
	slave_id = ((uint64_t)(uintptr_t)sdesc->hspi.Instance << 32) |
		   sdesc->chip_select->number;
	if (slave_id != last_slave_id) {
		last_slave_id = slave_id;
		ret = stm32_spi_config(desc);
		if (ret)
			return ret;
	}

	for (uint32_t i = 0; i < len; i++) {

		if (!msgs[i].tx_buff && !msgs[i].rx_buff)
			return -EINVAL;

		/* Assert CS */
		gdesc->port->BSRR = NO_OS_BIT(sdesc->chip_select->number) << 16;

		if (msgs[i].cs_delay_first)
			no_os_udelay(msgs[i].cs_delay_first);

#ifndef SPI_SR_TXE
		/* Some STM32 families have different naming for
		   SPI_SR_TXE, SPI_SR_RXNE and SPIx->DR. In that case, simply
		   use the HAL API for SPI transmission, which is generic
		   for all STM32 families. */

		if (msgs[i].tx_buff && msgs[i].rx_buff)
			ret = HAL_SPI_TransmitReceive(&sdesc->hspi, msgs[i].tx_buff, msgs[i].rx_buff,
						      msgs[i].bytes_number, HAL_MAX_DELAY);

		else if (msgs[i].tx_buff)
			ret = HAL_SPI_Transmit(&sdesc->hspi, msgs[i].tx_buff, msgs[i].bytes_number,
					       HAL_MAX_DELAY);
		else
			ret = HAL_SPI_Receive(&sdesc->hspi, msgs[i].rx_buff, msgs[i].bytes_number,
					      HAL_MAX_DELAY);

		if (ret != HAL_OK) {
			if (ret == HAL_TIMEOUT)
				ret = -ETIMEDOUT;
			else
				ret = -EIO;
		}
#else
		rx_cnt = 0;
		tx_cnt = 0;

		while ((msgs[i].rx_buff && rx_cnt < msgs[i].bytes_number) ||
		       (msgs[i].tx_buff && tx_cnt < msgs[i].bytes_number)) {
			while (!(SPIx->SR & SPI_SR_TXE))
				;
			if (msgs[i].tx_buff)
				*(volatile uint8_t *)&SPIx->DR = msgs[i].tx_buff[tx_cnt++];
			else
				*(volatile uint8_t *)&SPIx->DR = 0;

			while (!(SPIx->SR & SPI_SR_RXNE))
				;
			if (msgs[i].rx_buff)
				msgs[i].rx_buff[rx_cnt++] = *(volatile uint8_t *)&SPIx->DR;
			else
				(void)*(volatile uint8_t *)&SPIx->DR;
		}

#endif

		if (msgs[i].cs_delay_last)
			no_os_udelay(msgs[i].cs_delay_last);

		if (msgs[i].cs_change)
			/* De-assert CS */
			gdesc->port->BSRR = NO_OS_BIT(sdesc->chip_select->number);

		if (msgs[i].cs_change_delay)
			no_os_udelay(msgs[i].cs_change_delay);

		if (ret)
			return ret;
	}

	return 0;
}

/**
 * @brief Write and read data to/from SPI.
 * @param desc - The SPI descriptor.
 * @param data - The buffer with the transmitted/received data.
 * @param bytes_number - Number of bytes to write/read.
 * @return 0 in case of success, -1 otherwise.
 */
int32_t stm32_spi_write_and_read(struct no_os_spi_desc *desc,
				 uint8_t *data,
				 uint16_t bytes_number)
{
	struct no_os_spi_msg msg = {
		.bytes_number = bytes_number,
		.cs_change = true,
		.rx_buff = data,
		.tx_buff = data,
	};

	if (!desc || !desc->extra || !data)
		return -EINVAL;

	if (!bytes_number)
		return 0;

	return stm32_spi_transfer(desc, &msg, 1);
}

/**
 * @brief Configure and start a series of transfers using DMA.
 * @param desc - The SPI descriptor.
 * @param msgs - The messages array.
 * @param len - Number of messages.
 * @param callback - Function to be invoked after transfers
 * @param ctx - User defined parameter for the callback function.
 * @return 0 in case of success, errno codes otherwise.
 */
int32_t stm32_config_dma_and_start(struct no_os_spi_desc* desc,
				   struct no_os_spi_msg* msgs,
				   uint32_t len,
				   void (*callback)(
						   struct no_os_dma_xfer_desc *old_xfer,
						   struct no_os_dma_xfer_desc *next_xfer,
						   void *ctx),
				   void* ctx)
{
	struct stm32_spi_desc* sdesc = desc->extra;
	struct no_os_dma_xfer_desc* rx_ch_xfer;
	struct no_os_dma_xfer_desc* tx_ch_xfer;
	SPI_TypeDef* SPIx = sdesc->hspi.Instance;
	int ret;
	uint8_t i;

	if (!desc || !msgs)
		return -EINVAL;

	rx_ch_xfer = no_os_calloc(len, sizeof(*rx_ch_xfer));
	if (!rx_ch_xfer)
		return -ENOMEM;

	tx_ch_xfer = no_os_calloc(len, sizeof(*tx_ch_xfer));
	if (!tx_ch_xfer) {
		goto free_rx_ch_xfer;
	}

	for (i = 0; i < len; i++) {
		tx_ch_xfer[i].src = msgs[i].tx_buff;
#ifndef SPI_SR_TXE
		tx_ch_xfer[i].dst = &(SPIx->TXDR);
#else
		tx_ch_xfer[i].dst = &(SPIx->DR);
#endif
		tx_ch_xfer[i].xfer_type = MEM_TO_DEV;
		tx_ch_xfer[i].periph = NO_OS_DMA_IRQ;
		tx_ch_xfer[i].length = msgs[i].bytes_number;

		rx_ch_xfer[i].dst = msgs[i].rx_buff;
#ifndef SPI_SR_RXNE
		rx_ch_xfer[i].src = &(SPIx->RXDR);
#else
		rx_ch_xfer[i].src = &(SPIx->DR);
#endif
		rx_ch_xfer[i].periph = NO_OS_DMA_IRQ;
		rx_ch_xfer[i].xfer_type = DEV_TO_MEM;
		rx_ch_xfer[i].length = msgs[i].bytes_number;
		if (callback) {
			rx_ch_xfer[i].xfer_complete_cb = callback;
			rx_ch_xfer[i].xfer_complete_ctx = ctx;

			tx_ch_xfer[i].xfer_complete_cb = NULL;
			tx_ch_xfer[i].xfer_complete_ctx = NULL;
		}
	}

	sdesc->tx_ch_xfer = tx_ch_xfer;
	sdesc->rx_ch_xfer = rx_ch_xfer;

	ret = no_os_dma_config_xfer(sdesc->dma_desc, rx_ch_xfer, len, sdesc->rxdma_ch);
	if (ret)
		goto remove_dma;

	ret = no_os_dma_config_xfer(sdesc->dma_desc, tx_ch_xfer, len, sdesc->txdma_ch);
	if (ret)
		goto remove_dma;

	ret = no_os_dma_xfer_start(sdesc->dma_desc, sdesc->txdma_ch);
	if (ret)
		goto abort_transfer;

	ret = no_os_dma_xfer_start(sdesc->dma_desc, sdesc->rxdma_ch);
	if (ret)
		goto abort_transfer;

	if (sdesc->txdma_ch)
#if defined (STM32H5)
		SET_BIT(sdesc->hspi.Instance->CFG1, SPI_CFG1_TXDMAEN);
#else
		SET_BIT(sdesc->hspi.Instance->CR2, SPI_CR2_TXDMAEN);
#endif

	if (sdesc->rxdma_ch)
#if defined (STM32H5)
		SET_BIT(sdesc->hspi.Instance->CFG1, SPI_CFG1_RXDMAEN);
#else
		SET_BIT(sdesc->hspi.Instance->CR2, SPI_CR2_RXDMAEN);
#endif

#ifdef HAL_TIM_MODULE_ENABLED
	if (sdesc->pwm_desc) {
		stm32_spi_altrnate_cs_enable(desc, true);
		ret = no_os_pwm_enable(sdesc->pwm_desc);
		if (ret)
			goto abort_transfer;
	}
	if (sdesc->tx_pwm_desc) {
		ret = no_os_pwm_enable(sdesc->tx_pwm_desc);
		if (ret)
			goto abort_transfer;
	}
#endif

	return 0;

abort_transfer:
	no_os_dma_xfer_abort(sdesc->dma_desc, sdesc->txdma_ch);
	no_os_dma_xfer_abort(sdesc->dma_desc, sdesc->rxdma_ch);
remove_dma:
	no_os_dma_remove(sdesc->dma_desc);
free_tx_ch_xfer:
	no_os_free(tx_ch_xfer);
free_rx_ch_xfer:
	no_os_free(rx_ch_xfer);

	return ret;
}

void stm32_spi_dma_callback(struct no_os_dma_xfer_desc *old_xfer,
			    struct no_os_dma_xfer_desc *next_xfer,
			    void *ctx)
{
	struct no_os_spi_desc* desc = ctx;
	struct stm32_spi_desc* sdesc = desc->extra;
	SPI_TypeDef * SPIx = sdesc->hspi.Instance;

	/* if more xfers pending dont do anything */
	if (next_xfer)
		return;

#ifdef HAL_TIM_MODULE_ENABLED
	if (sdesc->pwm_desc)
		no_os_pwm_disable(sdesc->pwm_desc);

	if (sdesc->tx_pwm_desc)
		no_os_pwm_disable(sdesc->tx_pwm_desc);
#endif

	/* Perform abort SPI transfers */
	no_os_spi_transfer_abort(desc);

	/* Free the allocated memory for tx and rx transfers */
	no_os_free(sdesc->tx_ch_xfer);
	no_os_free(sdesc->rx_ch_xfer);

	/* put CS pin back into gpio mode */
	stm32_spi_altrnate_cs_enable(desc, false);

	sdesc->stm32_spi_dma_done = true;

	if (sdesc->stm32_spi_dma_user_cb)
		sdesc->stm32_spi_dma_user_cb(sdesc->stm32_spi_dma_user_ctx);
}

/**
 * @brief Configure and start a series of transfers using DMA. Don't Wait for the
 * 	  completion before returning.
 * @param desc - The SPI descriptor.
 * @param msgs - The messages array.
 * @param len - Number of messages.
 * @param callback - Function to be invoked once the transfers are done.
 * @param ctx - User defined parameter for the callback function.
 * @return 0 in case of success, errno codes otherwise.
 */
int32_t stm32_spi_transfer_dma_async(struct no_os_spi_desc* desc,
				     struct no_os_spi_msg* msgs,
				     uint32_t len,
				     void (*callback)(void*),
				     void* ctx)
{
	struct stm32_spi_desc* sdesc = desc->extra;

	sdesc->stm32_spi_dma_user_cb = callback;
	sdesc->stm32_spi_dma_user_ctx = ctx;
	return stm32_config_dma_and_start(desc, msgs, len, stm32_spi_dma_callback,
					  desc);
}

/**
 * @brief Configure and start a series of transfers using DMA. Wait for the
 * 	  completion before returning.
 * @param desc - The SPI descriptor.
 * @param msgs - The messages array.
 * @param len - Number of messages.
 * @return 0 in case of success, errno codes otherwise.
 */
int32_t stm32_spi_transfer_dma(struct no_os_spi_desc* desc,
			       struct no_os_spi_msg* msgs,
			       uint32_t len)
{
	uint32_t timeout;
	struct stm32_spi_desc* sdesc = desc->extra;

	sdesc->stm32_spi_dma_done = false;
	stm32_config_dma_and_start(desc, msgs, len, stm32_spi_dma_callback, desc);
	timeout = msgs->bytes_number;
	while (timeout--) {
		no_os_mdelay(1);
		if (sdesc->stm32_spi_dma_done)
			break;
	};

	/* need some cleanup here? */
	if (timeout == 0)
		return -ETIME;

	return 0;
}

/**
 * @brief Abort SPI transfers.
 * @param desc - The SPI descriptor.
 * @return 0 in case of success, errno codes otherwise.
 */
int32_t stm32_spi_transfer_abort(struct no_os_spi_desc* desc)
{
	int32_t ret;
	struct stm32_spi_desc* sdesc;

	if (!desc->extra)
		return -EINVAL;

	sdesc = desc->extra;
	SPI_TypeDef *SPIx = sdesc->hspi.Instance;

	if (sdesc->rxdma_ch) {
		ret = no_os_dma_xfer_abort(sdesc->dma_desc, sdesc->rxdma_ch);
		if (ret) {
			return ret;
		}

#if defined (STM32H5)
		CLEAR_BIT(SPIx->CFG1, SPI_CFG1_RXDMAEN);
#else
		CLEAR_BIT(SPIx->CR2, SPI_CR2_RXDMAEN);
#endif
	}

	if (sdesc->txdma_ch) {
		ret = no_os_dma_xfer_abort(sdesc->dma_desc, sdesc->txdma_ch);
		if (ret) {
			return ret;
		}

#if defined (STM32H5)
		CLEAR_BIT(SPIx->CFG1, SPI_CFG1_TXDMAEN);
#else
		CLEAR_BIT(SPIx->CR2, SPI_CR2_TXDMAEN);
#endif
	}

	/* Dummy read to clear any pending read on SPI */
#ifndef SPI_SR_RXNE
	*(volatile uint8_t *)&SPIx->RXDR;
#else
	*(volatile uint8_t *)&SPIx->DR;
#endif

	return 0;
}

/**
 * @brief stm32 platform specific SPI platform ops structure
 */
const struct no_os_spi_platform_ops stm32_spi_ops = {
	.init = &stm32_spi_init,
	.write_and_read = &stm32_spi_write_and_read,
	.remove = &stm32_spi_remove,
	.transfer = &stm32_spi_transfer,
	.transfer_dma_async = &stm32_spi_transfer_dma_async,
	.transfer_dma = &stm32_spi_transfer_dma,
	.transfer_abort = &stm32_spi_transfer_abort
};
