# Copyright (C) 2022 The Qt Company Ltd.
# SPDX-License-Identifier: BSD-3-Clause
#.rst:
# FindFFmpeg
# ----------
#
# Try to find the required ffmpeg components (default: AVFORMAT, AVUTIL, AVCODEC)
#
# Next variables can be used to hint FFmpeg libs search:
#
# ::
#
#   PC_<component>_LIBRARY_DIRS
#   PC_FFMPEG_LIBRARY_DIRS
#   PC_<component>_INCLUDE_DIRS
#   PC_FFMPEG_INCLUDE_DIRS
#
# Once done this will define
#
# ::
#
#   FFMPEG_FOUND         - System has the all required components.
#   FFMPEG_INCLUDE_DIRS  - Include directory necessary for using the required components headers.
#   FFMPEG_LIBRARIES     - Link these to use the required ffmpeg components.
#   FFMPEG_LIBRARY_DIRS  - Link directories
#   FFMPEG_DEFINITIONS   - Compiler switches required for using the required ffmpeg components.
#
# For each of the components it will additionally set.
#
# ::
#
#   AVCODEC
#   AVDEVICE
#   AVFORMAT
#   AVFILTER
#   AVUTIL
#   POSTPROC
#   SWSCALE
#
# the following variables will be defined
#
# ::
#
#   <component>_FOUND        - System has <component>
#   FFMPEG_<component>_FOUND - System has <component> (as checked by FHSPA)
#   <component>_INCLUDE_DIRS - Include directory necessary for using the <component> headers
#   <component>_LIBRARIES    - Link these to use <component>
#   <component>_LIBRARY_DIRS - Link directories
#   <component>_DEFINITIONS  - Compiler switches required for using <component>
#   <component>_VERSION      - The components version
#
# the following import targets is created
#
# ::
#
#   FFmpeg::FFmpeg - for all components
#   FFmpeg::<component> - where <component> in lower case (FFmpeg::avcodec) for each components
#
# Copyright (c) 2006, Matthias Kretz, <kretz@kde.org>
# Copyright (c) 2008, Alexander Neundorf, <neundorf@kde.org>
# Copyright (c) 2011, Michael Jansen, <kde@michael-jansen.biz>
# Copyright (c) 2017, Alexander Drozdov, <adrozdoff@gmail.com>
#

include(FindPackageHandleStandardArgs)

# The default components were taken from a survey over other FindFFMPEG.cmake files
if (NOT FFmpeg_FIND_COMPONENTS)
  set(FFmpeg_FIND_COMPONENTS AVCODEC AVFORMAT AVUTIL)
endif ()

if (QT_DEPLOY_FFMPEG AND BUILD_SHARED_LIBS)
    set(shared_libs_required TRUE)
endif()

#
### Macro: set_component_found
#
# Marks the given component as found if both *_LIBRARY_NAME AND *_INCLUDE_DIRS is present.
#
macro(set_component_found _component)
  if (${_component}_LIBRARY_NAME AND ${_component}_INCLUDE_DIR AND
      (${_component}_SHARED_LIBRARIES OR NOT shared_libs_required))
      # message(STATUS "  - ${_component} found.")
    set(${_component}_FOUND TRUE)
    set(${CMAKE_FIND_PACKAGE_NAME}_${_component}_FOUND TRUE)
  else ()
    # message(STATUS "  - ${_component} not found.")
  endif ()
endmacro()

find_package(PkgConfig QUIET)
if (NOT PKG_CONFIG_FOUND AND NOT FFMPEG_DIR)
    set(FFMPEG_DIR "/usr/local")
endif()
#
### Macro: find_component
#
# Checks for the given component by invoking pkgconfig and then looking up the libraries and
# include directories.
#
macro(find_component _component _pkgconfig _library _header)

  # use pkg-config to get the directories and then use these values
  # in the FIND_PATH() and FIND_LIBRARY() calls
  if (PKG_CONFIG_FOUND AND NOT FFMPEG_DIR)
    pkg_check_modules(PC_${_component} ${_pkgconfig})
  endif ()

  if (FFMPEG_DIR OR FFMPEG_ROOT)
      set(__find_ffmpeg_backup_root_dir "${CMAKE_FIND_ROOT_PATH}")
  endif()

  if(FFMPEG_DIR)
       list(APPEND CMAKE_FIND_ROOT_PATH "${FFMPEG_DIR}")
  endif()

  if(FFMPEG_ROOT)
      list(APPEND CMAKE_FIND_ROOT_PATH "${FFMPEG_ROOT}")
  endif()

  if (${_component}_INCLUDE_DIR AND NOT EXISTS ${${_component}_INCLUDE_DIR})
    message(STATUS "Cached include dir ${${_component}_INCLUDE_DIR} doesn't exist")
    unset(${_component}_INCLUDE_DIR CACHE)
  endif()

  find_path(${_component}_INCLUDE_DIR ${_header}
    HINTS
      ${PC_${_component}_INCLUDEDIR}
      ${PC_${_component}_INCLUDE_DIRS}
      ${PC_FFMPEG_INCLUDE_DIRS}
    PATHS
      ${FFMPEG_DIR}
    PATH_SUFFIXES
      ffmpeg include
  )

  if (shared_libs_required AND NOT WIN32)
    set(CMAKE_FIND_LIBRARY_SUFFIXES ${CMAKE_SHARED_LIBRARY_SUFFIX})
  endif()

  if (${_component}_LIBRARY AND NOT EXISTS ${${_component}_LIBRARY})
    message(STATUS "Cached library ${${_component}_LIBRARY} doesn't exist")
    unset(${_component}_LIBRARY CACHE)
  endif()

  find_library(${_component}_LIBRARY
    NAMES ${PC_${_component}_LIBRARIES} ${_library}
    HINTS
      ${PC_${_component}_LIBDIR}
      ${PC_${_component}_LIBRARY_DIRS}
      ${PC_FFMPEG_LIBRARY_DIRS}
    PATHS
      ${FFMPEG_DIR}
    PATH_SUFFIXES
      lib bin
  )

  if(FFMPEG_DIR OR FFMPEG_ROOT)
    set(CMAKE_FIND_ROOT_PATH "${__find_ffmpeg_backup_root_dir}")
  endif()

  if (${_component}_LIBRARY)
    get_filename_component(${_component}_LIBRARY_DIR ${${_component}_LIBRARY} DIRECTORY)
    get_filename_component(${_component}_LIBRARY_NAME ${${_component}_LIBRARY} NAME)

    # On Windows, shared linking goes through 'integration' static libs, so we should look for shared ones anyway
    # On Unix, we gather symlinks as well so that we could install them.
    if (WIN32 OR ${${_component}_LIBRARY_NAME} MATCHES "\\${CMAKE_SHARED_LIBRARY_SUFFIX}$")
      # the searching pattern is pretty rough but it seems to be sufficient to gather dynamic libs
       get_filename_component(name_we ${${_component}_LIBRARY} NAME_WE)

      file(GLOB ${_component}_SHARED_LIBRARIES "${${_component}_LIBRARY_DIR}/${name_we}*${CMAKE_SHARED_LIBRARY_SUFFIX}*")
    endif()

  endif()

  set(${_component}_DEFINITIONS  ${PC_${_component}_CFLAGS_OTHER})
  set_component_found(${_component})
endmacro()

# Clear the previously cached variables, because they are recomputed every time
# the Find script is included.
unset(FFMPEG_INCLUDE_DIRS)
unset(FFMPEG_LIBRARIES)
unset(FFMPEG_SHARED_LIBRARIES)
unset(FFMPEG_DEFINITIONS)
unset(FFMPEG_LIBRARY_DIRS)

# Check for components.
foreach (_component ${FFmpeg_FIND_COMPONENTS})
  string(TOLOWER ${_component} library)
  find_component(${_component} "lib${library}" ${library} "lib${library}/${library}.h")

  if (${_component}_FOUND)
    list(APPEND FFMPEG_LIBRARIES    ${${_component}_LIBRARY_NAME})
    list(APPEND FFMPEG_DEFINITIONS  ${${_component}_DEFINITIONS})
    list(APPEND FFMPEG_INCLUDE_DIRS ${${_component}_INCLUDE_DIR})
    list(APPEND FFMPEG_LIBRARY_DIRS ${${_component}_LIBRARY_DIR})

    if (${_component}_SHARED_LIBRARIES)
      list(APPEND FFMPEG_SHARED_LIBRARIES ${${_component}_SHARED_LIBRARIES})
      list(APPEND FFMPEG_SHARED_COMPONENTS ${_component})
    else()
      list(APPEND FFMPEG_STATIC_COMPONENTS ${_component})
    endif()
  endif()
endforeach()

if (NOT FFMPEG_SHARED_COMPONENTS AND (ANDROID OR LINUX))
  set(ENABLE_DYNAMIC_RESOLVE_OPENSSL_SYMBOLS TRUE CACHE INTERNAL "")
endif()

set(ENABLE_DYNAMIC_RESOLVE_VAAPI_SYMBOLS ${LINUX} CACHE INTERNAL "")

function(__try_add_dynamic_resolve_dependency dep added)
  set(added TRUE PARENT_SCOPE)

  if(ENABLE_DYNAMIC_RESOLVE_OPENSSL_SYMBOLS AND
      (${dep} STREQUAL "ssl" OR ${dep} STREQUAL "crypto"))
    set(DYNAMIC_RESOLVE_OPENSSL_SYMBOLS TRUE CACHE INTERNAL "")
  elseif(ENABLE_DYNAMIC_RESOLVE_VAAPI_SYMBOLS AND ${dep} STREQUAL "va")
    set(DYNAMIC_RESOLVE_VAAPI_SYMBOLS TRUE CACHE INTERNAL "")
  elseif(ENABLE_DYNAMIC_RESOLVE_VAAPI_SYMBOLS AND ${dep} STREQUAL "va-drm")
    set(DYNAMIC_RESOLVE_VA_DRM_SYMBOLS TRUE CACHE INTERNAL "")
  elseif(ENABLE_DYNAMIC_RESOLVE_VAAPI_SYMBOLS AND ${dep} STREQUAL "va-x11")
    set(DYNAMIC_RESOLVE_VA_X11_SYMBOLS TRUE CACHE INTERNAL "")
  else()
    set(added FALSE PARENT_SCOPE)
  endif()
endfunction()

# Function parses package config file to find the static library dependencies
# and adds them to the target library.
function(__ffmpeg_internal_set_dependencies lib)
  set(PC_FILE ${FFMPEG_DIR}/lib/pkgconfig/lib${lib}.pc)
  if(EXISTS ${PC_FILE})
    file(READ ${PC_FILE} pcfile)

    set(prefix_l "(^| )\\-l")
    set(suffix_lib "\\.lib($| )")

    string(REGEX REPLACE ".*Libs:([^\n\r]+).*" "\\1" out "${pcfile}")
    string(REGEX MATCHALL "${prefix_l}[^ ]+" libs_dependency ${out})
    string(REGEX MATCHALL "[^ ]+${suffix_lib}" libs_dependency_lib ${out})

    string(REGEX REPLACE ".*Libs.private:([^\n\r]+).*" "\\1" out "${pcfile}")
    string(REGEX MATCHALL "${prefix_l}[^ ]+" libs_private_dependency ${out})
    string(REGEX MATCHALL "[^ ]+${suffix_lib}" libs_private_dependency_lib ${out})

    list(APPEND deps_no_suffix ${libs_dependency} ${libs_private_dependency})
    foreach(dependency ${deps_no_suffix})
      string(REGEX REPLACE ${prefix_l} "" dependency ${dependency})
      if(NOT ${lib} STREQUAL ${dependency})
        __try_add_dynamic_resolve_dependency(${dependency} added)
        if(NOT added)
          target_link_libraries(FFmpeg::${lib} INTERFACE ${dependency})
        endif()
      endif()
    endforeach()

    list(APPEND deps_lib_suffix ${libs_dependency_lib} ${libs_private_dependency_lib})
    foreach(dependency ${deps_lib_suffix})
      string(REGEX REPLACE ${suffix_lib} "" dependency ${dependency})
      target_link_libraries(FFmpeg::${lib} INTERFACE ${dependency})
    endforeach()
  endif()
endfunction()

# Check for cached results. If there are skip the costly part.
#if (NOT FFMPEG_LIBRARIES)

  # Check if the required components were found and add their stuff to the FFMPEG_* vars.


  foreach (_component ${FFmpeg_FIND_COMPONENTS})
    if (${_component}_FOUND)
      string(TOLOWER ${_component} _lowerComponent)
      if (NOT TARGET FFmpeg::${_lowerComponent})
        add_library(FFmpeg::${_lowerComponent} INTERFACE IMPORTED)
        set_target_properties(FFmpeg::${_lowerComponent} PROPERTIES
            INTERFACE_COMPILE_OPTIONS "${${_component}_DEFINITIONS}"
            INTERFACE_INCLUDE_DIRECTORIES ${${_component}_INCLUDE_DIR}
            INTERFACE_LINK_LIBRARIES "${${_component}_LIBRARY_NAME}"
            INTERFACE_LINK_DIRECTORIES "${${_component}_LIBRARY_DIR}"
        )
        if(NOT ${_component}_SHARED_LIBRARIES)
          __ffmpeg_internal_set_dependencies(${_lowerComponent})
        endif()
        target_link_libraries(FFmpeg::${_lowerComponent} INTERFACE "${${_component}_LIBRARY_NAME}")
        if (UNIX AND NOT APPLE)
          target_link_options(FFmpeg::${_lowerComponent} INTERFACE  "-Wl,--exclude-libs=lib${_lowerComponent}")
        endif ()
      endif()
    endif()
  endforeach ()

  # Build the include path with duplicates removed.
  list(REMOVE_DUPLICATES FFMPEG_INCLUDE_DIRS)
  list(REMOVE_DUPLICATES FFMPEG_LIBRARY_DIRS)
  list(REMOVE_DUPLICATES FFMPEG_SHARED_LIBRARIES)

  # cache the vars.
  set(FFMPEG_INCLUDE_DIRS ${FFMPEG_INCLUDE_DIRS} CACHE STRING "The FFmpeg include directories." FORCE)
  set(FFMPEG_LIBRARIES    ${FFMPEG_LIBRARIES}    CACHE STRING "The FFmpeg libraries." FORCE)
  set(FFMPEG_DEFINITIONS  ${FFMPEG_DEFINITIONS}  CACHE STRING "The FFmpeg cflags." FORCE)
  set(FFMPEG_LIBRARY_DIRS ${FFMPEG_LIBRARY_DIRS} CACHE STRING "The FFmpeg library dirs." FORCE)
  set(FFMPEG_SHARED_LIBRARIES ${FFMPEG_SHARED_LIBRARIES} CACHE STRING "The FFmpeg dynamic libraries." FORCE)

  mark_as_advanced(FFMPEG_INCLUDE_DIRS
                   FFMPEG_LIBRARIES
                   FFMPEG_DEFINITIONS
                   FFMPEG_LIBRARY_DIRS
                   FFMPEG_SHARED_LIBRARIES
               )
# endif ()

list(LENGTH FFMPEG_LIBRARY_DIRS DIRS_COUNT)
if (${DIRS_COUNT} GREATER 1)
  message(WARNING "One ffmpeg library dir is expected, found dirs: ${FFMPEG_LIBRARY_DIRS}")
endif()

if(FFMPEG_SHARED_COMPONENTS AND FFMPEG_STATIC_COMPONENTS)
  message(WARNING
    "Only static or shared components are expected\n"
    "  static components: ${FFMPEG_STATIC_COMPONENTS}\n"
    "  static components: ${FFMPEG_SHARED_COMPONENTS}")
endif()

if (NOT TARGET FFmpeg::FFmpeg)
  add_library(FFmpeg INTERFACE)
  set_target_properties(FFmpeg PROPERTIES
      INTERFACE_COMPILE_OPTIONS "${FFMPEG_DEFINITIONS}"
      INTERFACE_INCLUDE_DIRECTORIES "${FFMPEG_INCLUDE_DIRS}"
      INTERFACE_LINK_LIBRARIES "${FFMPEG_LIBRARIES}"
      INTERFACE_LINK_DIRECTORIES "${FFMPEG_LIBRARY_DIRS}"
  )
  add_library(FFmpeg::FFmpeg ALIAS FFmpeg)
endif()

# Compile the list of required vars
set(_FFmpeg_REQUIRED_VARS FFMPEG_LIBRARIES FFMPEG_INCLUDE_DIRS)
foreach (_component ${FFmpeg_FIND_COMPONENTS})
  list(APPEND _FFmpeg_REQUIRED_VARS ${_component}_LIBRARY ${_component}_INCLUDE_DIR)
endforeach ()

# Give a nice error message if some of the required vars are missing.
find_package_handle_standard_args(FFmpeg
    REQUIRED_VARS ${_FFmpeg_REQUIRED_VARS}
    HANDLE_COMPONENTS
)
